<?php
/**
 * Automatic & manual update-checker for
 * “Organic Safelink 4 page Plugin”.
 *
 * Requires a valid licence row in the wp_org_wpcodequill table.
 *
 * @author  WPCodeQuill
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/* ── guard-constants (plugin can be bundled in themes, MU-plugins, etc.) ── */
if ( ! defined( 'OA_TEXT_DOMAIN'     ) ) define( 'OA_TEXT_DOMAIN',     'organic-safelink-4-page-plugin' );
if ( ! defined( 'OA_PLUGIN_FILE'     ) ) define( 'OA_PLUGIN_FILE',     'organic-safelink-4-page-plugin/organic-safelink-4-page-plugin.php' );
if ( ! defined( 'OA_UPDATE_ENDPOINT' ) ) define( 'OA_UPDATE_ENDPOINT', 'https://admin.wpcodequill.com/my-admin/update/update-info.php' );

/* ---------------------------------------------------------------------------
 * 1.  Inject update data into the site-transient
 * ------------------------------------------------------------------------ */
add_filter( 'pre_set_site_transient_update_plugins', function ( $transient ) {

	if ( empty( $transient->checked ) ) {
		return $transient;                       // nothing to do
	}

	global $wpdb;
	$table  = $wpdb->prefix . 'org_wpcodequill';
	$domain = strtolower( preg_replace( '/^www\./i', '', $_SERVER['HTTP_HOST'] ) );

	$license = $wpdb->get_var(
		$wpdb->prepare(
			"SELECT `key`
			   FROM {$table}
			  WHERE `domain` = %s
			    AND `plugin` = %s
			  LIMIT 1",
			$domain,
			OA_TEXT_DOMAIN
		)
	);

	if ( ! $license ) {
		return $transient;                       // not licenced → no update
	}

	/* call the endpoint */
	$url = add_query_arg(
		[
			'license' => $license,
			'domain'  => $domain,
			'plugin'  => OA_TEXT_DOMAIN,
		],
		OA_UPDATE_ENDPOINT
	);

	$response = wp_remote_get( $url, [ 'timeout' => 15 ] );
	if ( is_wp_error( $response ) ) {
		return $transient;
	}

	$data = json_decode( wp_remote_retrieve_body( $response ), true );
	if ( ! is_array( $data ) || empty( $data['version'] ) || empty( $data['download_url'] ) ) {
		return $transient;
	}

	$current = $transient->checked[ OA_PLUGIN_FILE ] ?? '0.0.0';

	if ( version_compare( $data['version'], $current, '>' ) ) {

		$transient->response[ OA_PLUGIN_FILE ] = (object) [
			'slug'         => $data['slug'],
			'plugin'       => OA_PLUGIN_FILE,
			'new_version'  => $data['version'],
			'package'      => $data['download_url'],
			'tested'       => $data['tested']       ?? '',
			'requires'     => $data['requires']     ?? '',
			'last_updated' => $data['last_updated'] ?? '',
			'sections'     => $data['sections']     ?? [],
			'banners'      => $data['banners']      ?? [],
			'icons'        => $data['icons']        ?? [],
			'author'       => $data['author']       ?? '',
			'homepage'     => $data['homepage']     ?? '',
		];
	}

	return $transient;
} );

/* ---------------------------------------------------------------------------
 * 2.  Manual “Check for updates” link (appears under the plugin name)
 * ------------------------------------------------------------------------ */
add_filter( 'plugin_action_links_' . OA_PLUGIN_FILE, function ( $links ) {

	$url     = wp_nonce_url( admin_url( 'plugins.php?oa_check_updates=1' ), 'oa_force_check' );
	$links[] = '<a href="' . esc_url( $url ) . '">' . esc_html__( 'Check for updates', 'default' ) . '</a>';

	return $links;
} );

/* ---------------------------------------------------------------------------
 * 3.  Handle the manual check
 * ------------------------------------------------------------------------ */
add_action( 'admin_init', function () {

	if ( isset( $_GET['oa_check_updates'] ) && current_user_can( 'update_plugins' ) ) {
		check_admin_referer( 'oa_force_check' );
		delete_site_transient( 'update_plugins' );   // clear cache
		wp_update_plugins();                         // force re-fetch
		wp_safe_redirect( admin_url( 'plugins.php' ) );
		exit;
	}
} );
